"""Module for custom extensions of the pandas Indexer classes"""

from pandas.core.indexers.objects import BaseIndexer
import numpy as np 

class FixedOffsetForwardIndexer(BaseIndexer):
    """Similar to FixedForwardIndexer, but allows the user to input an 'offset' parameter.
    The forward sliding window will start from [offset] number of rows ahead of the current row"""

    def __init__(
        self,
        index_array: np.ndarray | None = None,
        window_size: int = 0,
        index=None,
        offset=None,
        **kwargs,
    ) -> None:
        super().__init__(index_array, window_size, **kwargs)
        self.offset = offset

    def get_window_bounds(
        self,
        num_values: int = 0,
        min_periods: int | None = None,
        center: bool | None = None,
        closed: str | None = None,
        step: int | None = None,
    ) -> tuple[np.ndarray, np.ndarray]:

        if center:
            raise ValueError("Forward-looking windows can't have center=True")
        if closed is not None:
            raise ValueError(
                "Forward-looking windows don't support setting the closed argument"
            )
        if step is None:
            step = 1

        start = np.arange(0, num_values, step, dtype="int64") + self.offset
        end = start + self.window_size
        if self.window_size:
            end = np.clip(end, 0, num_values)

        return start, end